<?php

function detectLanguage(): string {
    $uri = parse_url($_SERVER['REQUEST_URI'] ?? '/', PHP_URL_PATH);
    // Normalize double slashes
    $uri = preg_replace('#/+/#', '/', $uri);
    if (preg_match('#/(de)(/|$)#', $uri)) {
        return 'de';
    }
    if (preg_match('#/(en)(/|$)#', $uri)) {
        return 'en';
    }
    // Default to German when no explicit language segment is present
    return 'de';
}

function getLocalizedContent(): array {
    $lang = detectLanguage();
    $file = __DIR__ . '/../content.' . $lang . '.json';
    if (!file_exists($file)) {
        $file = __DIR__ . '/../content.en.json'; // Fallback to English
    }
    $json = file_get_contents($file);
    $data = json_decode($json, true) ?: [];
    $data['_ui'] = getUiStrings($lang);
    return $data;
}

function getUiStrings(string $lang): array {
    $de = [
        'home' => 'Home',
        'services' => 'Dienstleistungen',
        'robots' => 'Roboter',
        'contact' => 'Kontakt',
        'about' => 'Über',
        'slogan_main' => 'Moving your process.',
        'slogan_smarter' => 'Smarter.',
        'imprint' => 'Impressum',
        'privacy' => 'Datenschutz',
        'environments' => 'Einsatzbereiche',
        // Optional UI strings for call features
        'or_call' => 'Oder direkt anrufen:',
        'call_now' => 'Jetzt anrufen',
        // Mobile phone for call features
        'phone_mobile' => '+4917674707142'
    ];
    $en = [
        'home' => 'Home',
        'services' => 'Services',
        'robots' => 'Robots',
        'contact' => 'Contact',
        'about' => 'About',
        'slogan_main' => 'Moving your process.',
        'slogan_smarter' => 'Smarter.',
        'imprint' => 'Imprint',
        'privacy' => 'Privacy Policy',
        'environments' => 'Industries',
        'or_call' => 'Or call directly:',
        'call_now' => 'Call now',
        'phone_mobile' => '+4917674707142'
    ];
    return $lang === 'de' ? $de : $en;
}

function basePath(): string {
    $docRoot = rtrim($_SERVER['DOCUMENT_ROOT'] ?? '', '/');
    $projectRoot = rtrim(dirname(__DIR__), '/'); // This will be /var/www/html/frisilogix

    if (strpos($projectRoot, $docRoot) === 0) {
        $base = substr($projectRoot, strlen($docRoot));
        return $base;
    }
    return ''; // If project root is not under document root, or other issues
}

function switchUrls(): array {
    $base = basePath();
    $uri = parse_url($_SERVER['REQUEST_URI'] ?? '/', PHP_URL_PATH);
    // Remove base prefix if present
    if ($base !== '' && strpos($uri, $base) === 0) {
        $tail = substr($uri, strlen($base));
    } else {
        $tail = $uri;
    }
    // Strip existing language segment from the tail
    $tail = preg_replace('#^/(de|en)#', '', $tail);
    if ($tail === '') { $tail = '/'; }

    // Translate file names between languages
    $deTail = translateTailForLang($tail, 'de');
    $enTail = translateTailForLang($tail, 'en');

    return [
        'de' => ($base ?: '') . '/de' . $deTail,
        'en' => ($base ?: '') . '/en' . $enTail,
    ];
}

/**
 * Translate the tail path (without base and language) to the target language.
 * Keeps directories and anchors conceptually; focuses on file name mapping.
 */
function translateTailForLang(string $tail, string $toLang): string {
    // Normalize
    $tail = '/' . ltrim($tail, '/');
    // Collapse double slashes
    $tail = preg_replace('#/+#', '/', $tail);
    // Map known filenames
    $mapDeToEn = [
        '/impressum.php'   => '/imprint.php',
        '/datenschutz.php' => '/privacy.php',
        '/ueber.php'  => '/about.php',
        '/danke.php'       => '/thanks.php',
        '/index.php'       => '/',
    ];
    $mapEnToDe = [
        '/imprint.php' => '/impressum.php',
        '/privacy.php' => '/datenschutz.php',
        '/about.php'   => '/ueber.php',
        '/thanks.php'  => '/danke.php',
        '/index.php'   => '/',
    ];

    // Extract only the filename part for mapping; preserve directories if any
    $dir = rtrim(dirname($tail), '/');
    $file = basename($tail);
    if ($tail === '/' || $tail === '') {
        $file = '';
        $dir = '';
    }

    if ($toLang === 'en') {
        $candidate = '/' . $file;
        if (isset($mapDeToEn[$candidate])) {
            $mapped = $mapDeToEn[$candidate];
        } else {
            $mapped = $candidate;
        }
    } else { // 'de'
        $candidate = '/' . $file;
        if (isset($mapEnToDe[$candidate])) {
            $mapped = $mapEnToDe[$candidate];
        } else {
            $mapped = $candidate;
        }
    }

    // Recombine with directory if there was one (rare in this site)
    if ($dir !== '' && $dir !== '.' && $mapped !== '/') {
        // Avoid double slashes when dir is root
        $mapped = rtrim($dir, '/') . $mapped;
    }

    // Ensure leading slash
    $mapped = '/' . ltrim($mapped, '/');
    // Normalize index to '/'
    if ($mapped === '/index.php') { $mapped = '/'; }
    return $mapped;
}

function hreflangLinks(string $lang, string $path): string {
    $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') ? 'https://' : 'http://';
    $host = $_SERVER['HTTP_HOST'] ?? '';
    $base = basePath();
    // Ensure path begins with a single slash and is a normalized tail
    $path = '/' . ltrim($path, '/');
    $deTail = translateTailForLang($path, 'de');
    $enTail = translateTailForLang($path, 'en');
    $deHref = $scheme . $host . ($base ?: '') . '/de' . $deTail;
    $enHref = $scheme . $host . ($base ?: '') . '/en' . $enTail;
    $lines = [];
    $lines[] = '<link rel="alternate" hreflang="de" href="' . htmlspecialchars($deHref, ENT_QUOTES) . '" />';
    $lines[] = '<link rel="alternate" hreflang="en" href="' . htmlspecialchars($enHref, ENT_QUOTES) . '" />';
    // x-default points to English by default
    $lines[] = '<link rel="alternate" hreflang="x-default" href="' . htmlspecialchars($enHref, ENT_QUOTES) . '" />';
    return implode("\n    ", $lines);
}

function relativePathWithoutLang(): string {
    $base = basePath();
    $uri = parse_url($_SERVER['REQUEST_URI'] ?? '/', PHP_URL_PATH) ?: '/';
    // Remove base prefix
    if ($base !== '' && strpos($uri, $base) === 0) {
        $uri = substr($uri, strlen($base));
    }
    // Strip language prefix
    $uri = preg_replace('#^/(de|en)(?=/|$)#', '', $uri);
    // Ensure non-empty path with leading slash
    if ($uri === '' || $uri[0] !== '/') {
        $uri = '/' . ltrim($uri, '/');
    }
    return $uri;
}



function isMobile(): bool {
    $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';
    $mobileAgents = [
        '/Mobile/i', '/' . preg_quote('Android') . '/i', '/' . preg_quote('iPhone') . '/i', '/' . preg_quote('iPad') . '/i',
        '/' . preg_quote('BlackBerry') . '/i', '/' . preg_quote('Windows Phone') . '/i', '/' . preg_quote('Opera Mini') . '/i',
        '/' . preg_quote('IEMobile') . '/i', '/' . preg_quote('webOS') . '/i', '/' . preg_quote('Kindle') . '/i'
    ];
    foreach ($mobileAgents as $agent) {
        if (preg_match($agent, $userAgent)) {
            return true;
        }
    }
    return false;
}
