<?php
// Minimal SMTP mailer (AUTH LOGIN + STARTTLS/SSL) without external deps.
// Optimized for reliable headers and delivery on common shared hosts.

function smtp_send(array $cfg, string $fromEmail, string $fromName, array $to, string $subject, string $body, array $headers = []): array {
    $host     = (string)($cfg['host']   ?? '');
    $port     = (int)   ($cfg['port']   ?? 465);
    $user     = (string)($cfg['user']   ?? '');
    $pass     = (string)($cfg['pass']   ?? '');
    $secure   = strtolower((string)($cfg['secure'] ?? 'ssl')); // ssl|tls|none
    $timeout  = (int)   ($cfg['timeout'] ?? 15);
    $debug    = (bool)  ($cfg['debug'] ?? false);
    $ehloHost = (string)($cfg['ehlo'] ?? 'localhost');
    $msgidDom = (string)($cfg['msgid_host'] ?? 'localhost');

    $debugFile = __DIR__ . '/../logs/smtp_debug.log';
    $dbg = function(string $line) use ($debug, $debugFile) {
        if (!$debug) return;
        if (!is_dir(dirname($debugFile))) { @mkdir(dirname($debugFile), 0775, true); }
        @file_put_contents($debugFile, '[' . date('c') . "] " . $line . "\n", FILE_APPEND);
    };

    if ($host === '' || empty($to)) {
        return [false, 'smtp_send: missing host or recipients'];
    }

    $transport = ($secure === 'ssl') ? 'ssl://' . $host : $host;

    $contextOptions = [];
    if ($secure !== 'none') {
        $contextOptions['ssl'] = [
            'SNI_enabled'       => true,
            'verify_peer'       => true,
            'verify_peer_name'  => true,
            'allow_self_signed' => false,
            'peer_name'         => $host,
        ];
    }
    $context = stream_context_create($contextOptions);

    $fp = @stream_socket_client($transport . ':' . $port, $errno, $errstr, $timeout, STREAM_CLIENT_CONNECT, $context);
    if (!$fp) return [false, "smtp_connect_failed: $errno $errstr"];
    stream_set_timeout($fp, $timeout);

    $read = function() use ($fp) {
        $resp = '';
        while (!feof($fp)) {
            $line = fgets($fp, 515);
            if ($line === false) break;
            $resp .= $line;
            if (isset($line[3]) && $line[3] === ' ') break; // end of multi-line
        }
        return $resp;
    };
    $send = function(string $cmd) use ($fp, $read) {
        fwrite($fp, $cmd . "\r\n");
        return $read();
    };

    // Server greeting
    $resp = $read();
    $dbg('S: ' . trim($resp));
    if (strpos($resp, '220') !== 0) return [false, 'smtp_greeting: ' . trim($resp)];

    // EHLO / HELO
    $dbg('C: EHLO ' . $ehloHost);
    $resp = $send('EHLO ' . $ehloHost);
    $dbg('S: ' . trim($resp));
    if (strpos($resp, '250') !== 0) {
        $dbg('C: HELO ' . $ehloHost);
        $resp = $send('HELO ' . $ehloHost);
        $dbg('S: ' . trim($resp));
        if (strpos($resp, '250') !== 0) return [false, 'smtp_ehlo_helo_failed: ' . trim($resp)];
    }

    // STARTTLS path (only if requested)
    if ($secure === 'tls') {
        $dbg('C: STARTTLS');
        $resp = $send('STARTTLS');
        $dbg('S: ' . trim($resp));
        if (strpos($resp, '220') !== 0) return [false, 'smtp_starttls_failed: ' . trim($resp)];

        $methods = [];
        if (defined('STREAM_CRYPTO_METHOD_TLSv1_3_CLIENT')) $methods[] = STREAM_CRYPTO_METHOD_TLSv1_3_CLIENT;
        if (defined('STREAM_CRYPTO_METHOD_TLSv1_2_CLIENT')) $methods[] = STREAM_CRYPTO_METHOD_TLSv1_2_CLIENT;
        if (defined('STREAM_CRYPTO_METHOD_TLSv1_1_CLIENT')) $methods[] = STREAM_CRYPTO_METHOD_TLSv1_1_CLIENT;
        if (defined('STREAM_CRYPTO_METHOD_TLSv1_0_CLIENT')) $methods[] = STREAM_CRYPTO_METHOD_TLSv1_0_CLIENT;
        if (defined('STREAM_CRYPTO_METHOD_TLS_CLIENT'))      $methods[] = STREAM_CRYPTO_METHOD_TLS_CLIENT;
        if (defined('STREAM_CRYPTO_METHOD_SSLv23_CLIENT'))   $methods[] = STREAM_CRYPTO_METHOD_SSLv23_CLIENT;

        $tlsOk = false;
        foreach ($methods as $m) {
            $dbg('TLS try method=' . $m);
            $tlsOk = @stream_socket_enable_crypto($fp, true, $m);
            if ($tlsOk) break;
        }
        if (!$tlsOk) {
            $opensslLoaded = extension_loaded('openssl') ? 'yes' : 'no';
            return [false, 'smtp_tls_enable_failed (openssl=' . $opensslLoaded . ')'];
        }

        // EHLO again after STARTTLS
        $dbg('C: EHLO ' . $ehloHost);
        $resp = $send('EHLO ' . $ehloHost);
        $dbg('S: ' . trim($resp));
        if (strpos($resp, '250') !== 0) return [false, 'smtp_ehlo_after_tls_failed: ' . trim($resp)];
    }

    // AUTH
    if ($user !== '') {
        // Prefer AUTH PLAIN then fallback to LOGIN
        $plainToken = base64_encode("\0" . $user . "\0" . $pass);
        $dbg('C: AUTH PLAIN <token>');
        $resp = $send('AUTH PLAIN ' . $plainToken);
        $dbg('S: ' . trim($resp));
        if (strpos($resp, '235') !== 0) {
            $dbg('C: AUTH LOGIN');
            $resp = $send('AUTH LOGIN');
            $dbg('S: ' . trim($resp));
            if (strpos($resp, '334') !== 0) return [false, 'smtp_auth_methods_not_accepted: ' . trim($resp)];

            $dbg('C: <username base64>');
            $resp = $send(base64_encode($user));
            $dbg('S: ' . trim($resp));
            if (strpos($resp, '334') !== 0) return [false, 'smtp_auth_username_rejected: ' . trim($resp)];

            $dbg('C: <password base64>');
            $resp = $send(base64_encode($pass));
            $dbg('S: ' . trim($resp));
            if (strpos($resp, '235') !== 0) return [false, 'smtp_auth_password_rejected: ' . trim($resp)];
        }
    }

    // Envelope sender (Return-Path)
    $mailFrom = (string)($cfg['bounce'] ?? $fromEmail);
    $dbg('C: MAIL FROM: <' . $mailFrom . '>');
    $resp = $send('MAIL FROM: <' . $mailFrom . '>');
    $dbg('S: ' . trim($resp));
    if (strpos($resp, '250') !== 0) return [false, 'smtp_mail_from_rejected: ' . trim($resp)];

    foreach ($to as $rcpt) {
        $dbg('C: RCPT TO: <' . $rcpt . '>');
        $resp = $send('RCPT TO: <' . $rcpt . '>');
        $dbg('S: ' . trim($resp));
        if (strpos($resp, '250') !== 0 && strpos($resp, '251') !== 0) {
            return [false, 'smtp_rcpt_to_rejected: ' . $rcpt . ' ' . trim($resp)];
        }
    }

    $dbg('C: DATA');
    $resp = $send('DATA');
    $dbg('S: ' . trim($resp));
    if (strpos($resp, '354') !== 0) return [false, 'smtp_data_not_accepted: ' . trim($resp)];

    // Build message
    $date = date('r');
    $msgidDom = preg_replace('/\s+/', '', $msgidDom) ?: 'localhost';
    $msgId = sprintf('<%s.%s@%s>', time(), bin2hex(random_bytes(6)), $msgidDom);

    $hdrLines = [];
    $hdrLines[] = 'Date: ' . $date;
    $hdrLines[] = 'Message-ID: ' . $msgId;
    $hdrLines[] = 'From: ' . (mime_header_encode($fromName) . ' <' . $fromEmail . '>');
    $hdrLines[] = 'To: ' . implode(', ', array_map(fn($e) => '<' . $e . '>', $to));
    $hdrLines[] = 'Subject: ' . mime_header_encode($subject);
    $hdrLines[] = 'MIME-Version: 1.0';
    $hdrLines[] = 'Content-Type: text/plain; charset=UTF-8';
    $hdrLines[] = 'Content-Transfer-Encoding: 8bit';

    foreach ($headers as $k => $v) {
        if ($k && $v) $hdrLines[] = trim($k) . ': ' . trim($v);
    }

    $data = implode("\r\n", $hdrLines) . "\r\n\r\n" . dot_escape($body) . "\r\n.";
    fwrite($fp, $data . "\r\n");
    $resp = $read();
    $dbg('S: ' . trim($resp));
    if (strpos($resp, '250') !== 0) return [false, 'smtp_data_termination_failed: ' . trim($resp)];

    $dbg('C: QUIT');
    $send('QUIT');
    fclose($fp);
    return [true, null];
}

// Encodes header token in RFC2047 Base64-UTF8 (safe for non-ASCII)
function mime_header_encode(string $text): string {
    if ($text === '') return '';
    $text = str_replace(["\r", "\n"], ' ', $text);
    if (preg_match('/^[\x20-\x7E]+$/', $text)) {
        return $text;
    }
    return '=?UTF-8?B?' . base64_encode($text) . '?=';
}

// Escape lines starting with "." according to SMTP DATA rules
function dot_escape(string $body): string {
    $body = str_replace(["\r\n", "\r"], "\n", $body);
    $lines = explode("\n", $body);
    foreach ($lines as &$line) {
        if ($line !== '' && $line[0] === '.') $line = '.' . $line;
    }
    return implode("\r\n", $lines);
}
