// js/contactForm.js

export function initContactForm() {
  const contactForm = document.getElementById('contact-form');
  const formStatus = document.getElementById('form-status');
  if (contactForm && formStatus) {
    contactForm.addEventListener('submit', function(e) {
      e.preventDefault();
      formStatus.textContent = '';
      formStatus.className = '';
      const formData = new FormData(contactForm);
      const base = (function() {
        const path = window.location.pathname;
        const m = path.match(/^(.*?)(?:\/(?:de|en))(?:\/|$)/);
        return m ? (m[1] || '/') : '/';
      })();
      const lang = (function() {
        const m = (window.location.pathname || '').match(/\/(de|en)(?:\/|$)/);
        return m ? m[1] : 'de';
      })();
      const MSG = {
        de: {
          sendError: 'Fehler beim Senden.',
          generic: 'Ein unerwarteter Fehler ist aufgetreten.'
        },
        en: {
          sendError: 'Error sending message.',
          generic: 'An unexpected error occurred.'
        }
      };
      // Validation messages
      MSG.de.validation = 'Bitte füllen Sie alle Felder korrekt aus und geben Sie eine gültige E-Mail an.';
      MSG.en.validation = 'Please complete all fields correctly and provide a valid email address.';

      // Basic client-side validation: required fields + email format
      const requiredFields = Array.from(contactForm.querySelectorAll('[required]'));
      let invalid = requiredFields.some(el => {
        const type = (el.type || '').toLowerCase();
        if (type === 'checkbox') {
          return !el.checked;
        }
        const v = (el.value || '').trim();
        if (!v) return true;
        if (type === 'email') {
          // simple email check
          return !/^[^ -()<>[]:;@\". ,\x00-\x1F\x7F]/.test(v);
        }
        return false;
      });
      // Fallback: re-evaluate with native validity and simpler email check
      if (invalid) {
        invalid = requiredFields.some(el => {
          if (!el.checkValidity()) return true;
          const type = (el.type || '').toLowerCase();
          if (type === 'email') {
            const v = (el.value || '').trim();
            return !/^[^@\s]+@[^@\s]+\.[^@\s]+$/.test(v);
          }
          return false;
        });
      }
      if (invalid) {
        formStatus.textContent = MSG[lang].validation;
        formStatus.className = 'error show';
        setTimeout(() => {
          formStatus.className = '';
          formStatus.textContent = '';
        }, 5000);
        return;
      }

      const postUrl = (base.endsWith('/') ? base.slice(0, -1) : base) + '/send_mail.php';
      fetch(postUrl, {
        method: 'POST',
        body: formData
      }).then(response => {
        if (response.ok) {
          const successPage = lang === 'en' ? 'thanks.php' : 'danke.php';
          const successUrl = (base || '') + '/' + lang + '/' + successPage;
          window.location.href = successUrl;
        } else {
          return response.text().then(text => {
            throw new Error(text || MSG[lang].sendError);
          });
        }
      }).catch(error => {
        formStatus.textContent = error.message || MSG[lang].generic;
        formStatus.className = 'error show';
        setTimeout(() => {
          formStatus.className = '';
          formStatus.textContent = '';
        }, 5000);
      });
    });
  }

  const openContactButtons = document.querySelectorAll('#openContact, #openContactHero, a[href="#contactModal"]');
  const closeContact = document.getElementById('closeContact');
  const contactModal = document.getElementById('contactModal');
  if (closeContact && contactModal) {
    const openModal = () => {
      contactModal.style.display = 'flex';
      document.body.classList.add('no-scroll');
      setTimeout(() => {
        contactModal.querySelector('input, textarea, button').focus();
      }, 100);
    };
    const closeModal = () => {
      contactModal.style.display = 'none';
      document.body.classList.remove('no-scroll');
    };
    openContactButtons.forEach(btn => btn.addEventListener('click', openModal));
    closeContact.addEventListener('click', closeModal);
    document.addEventListener('keydown', (e) => {
      if (e.key === 'Escape' && contactModal.style.display === 'flex') {
        closeModal();
      }
    });
    contactModal.addEventListener('click', (e) => {
      if (e.target === contactModal) {
        closeModal();
      }
    });
  }
}
