// js/cookieConsent.js

const COOKIE_NAME = 'cookie_consent';

// --- Analytics Placeholder --- //
// Add your analytics scripts inside this function.
// It will only be called if the user gives consent.
function loadAnalytics() {
  console.log('Analytics loaded. (Placeholder)');
  /*
  // Example for Google Analytics:
  const script = document.createElement('script');
  script.src = 'https://www.googletagmanager.com/gtag/js?id=DEINE_TRACKING_ID';
  script.async = true;
  document.head.appendChild(script);

  window.dataLayer = window.dataLayer || [];
  function gtag(){dataLayer.push(arguments);}
  gtag('js', new Date());
  gtag('config', 'DEINE_TRACKING_ID', { 'anonymize_ip': true });
  */
}

function getCookie(name) {
  const value = `; ${document.cookie}`;
  const parts = value.split(`; ${name}=`);
  if (parts.length === 2) return parts.pop().split(';').shift();
  return null;
}

function setCookie(name, value, days) {
  let expires = "";
  if (days) {
    const date = new Date();
    date.setTime(date.getTime() + (days * 24 * 60 * 60 * 1000));
    expires = "; expires=" + date.toUTCString();
  }
  document.cookie = name + "=" + (value || "") + expires + "; path=/; SameSite=Lax";
}

export function initCookieConsent() {
  const banner = document.getElementById('cookie-consent-banner');
  const acceptBtn = document.getElementById('cookie-accept');
  const declineBtn = document.getElementById('cookie-decline');

  if (!banner || !acceptBtn || !declineBtn) {
    return;
  }

  const consent = getCookie(COOKIE_NAME);

  if (consent === 'true') {
    document.body.classList.remove('cookie-pending');
    loadAnalytics();
  } else if (consent === null) {
    banner.classList.add('show');
    document.body.classList.add('cookie-pending');
    // move focus to accept for quick keyboard access
    try { setTimeout(() => acceptBtn.focus(), 0); } catch (_) {}
  }

  acceptBtn.addEventListener('click', () => {
    setCookie(COOKIE_NAME, 'true', 365);
    banner.classList.remove('show');
    document.body.classList.remove('cookie-pending');
    loadAnalytics();
  });

  declineBtn.addEventListener('click', () => {
    setCookie(COOKIE_NAME, 'false', 365);
    banner.classList.remove('show');
    document.body.classList.remove('cookie-pending');
  });
}

// Returns a promise that resolves once the user has made a cookie decision
// (accept or decline). If a decision already exists or no banner is present,
// it resolves immediately.
export function awaitCookieDecision() {
  return new Promise((resolve) => {
    const banner = document.getElementById('cookie-consent-banner');
    const acceptBtn = document.getElementById('cookie-accept');
    const declineBtn = document.getElementById('cookie-decline');
    const consent = getCookie(COOKIE_NAME);

    // If decision already stored or required elements are missing, resolve now
    if (consent !== null || !banner || !acceptBtn || !declineBtn) {
      resolve(consent);
      return;
    }

    const onResolve = () => {
      // Small microtask delay to ensure cookie is set by initCookieConsent handler
      setTimeout(() => resolve(getCookie(COOKIE_NAME)), 0);
      // Clean up one-time listeners
      acceptBtn.removeEventListener('click', onResolve);
      declineBtn.removeEventListener('click', onResolve);
    };

    acceptBtn.addEventListener('click', onResolve, { once: true });
    declineBtn.addEventListener('click', onResolve, { once: true });
  });
}
