// js/features.js

export function initFeatures() {
  // Auto-focus the feature closest to viewport center by toggling .is-focused
  (() => {
    const SELECTOR = '.feature.visible.in-view';
    let scheduled = false;
    let lastSwitchAt = 0; // timestamp to add small time hysteresis

    const applyFocus = () => {
      scheduled = false;
      // Honor short-lived user-forced focus window
      try {
        if (window.__featureUserFocusUntil && Date.now() < window.__featureUserFocusUntil) {
          return;
        }
      } catch (_) {}
      const nodes = Array.from(document.querySelectorAll(SELECTOR));

      // If the features section itself is out of view, clear focus and exit
      try {
        const wrap = document.querySelector('.features-wrap');
        if (wrap) {
          const vh = window.innerHeight || document.documentElement.clientHeight || 0;
          const midY = vh / 2;
          const wr = wrap.getBoundingClientRect();
          if (midY < wr.top || midY > wr.bottom) {
            document.querySelectorAll('.feature.is-focused').forEach(n => n.classList.remove('is-focused'));
            return;
          }
        }
      } catch (_) {}
      // Clear if none
      if (!nodes.length) {
        document.querySelectorAll('.feature.is-focused').forEach(n => n.classList.remove('is-focused'));
        return;
      }
      const vh = window.innerHeight || document.documentElement.clientHeight || 0;
      const midY = vh / 2;
      let best = null;
      let bestDist = Infinity;
      for (const el of nodes) {
        const r = el.getBoundingClientRect();
        // Only consider elements that intersect viewport at least partially
        if (r.bottom <= 0 || r.top >= vh) continue;
        const centerY = r.top + r.height / 2;
        const dist = Math.abs(centerY - midY);
        if (dist < bestDist) {
          best = el;
          bestDist = dist;
        }
      }
      // Update class only if needed
      const current = document.querySelector('.feature.is-focused');

      // Apply spatial + temporal hysteresis to avoid rapid toggling between neighbors
      if (current && best && best !== current) {
        const cr = current.getBoundingClientRect();
        const br = best.getBoundingClientRect();
        const cDist = Math.abs((cr.top + cr.height / 2) - midY);
        const bDist = Math.abs((br.top + br.height / 2) - midY);
        const HYST_PX = 24; // require at least 24px better to switch
        const HYST_MS = 140; // or wait a short time between switches
        const now = Date.now();
        const recentlySwitched = (now - lastSwitchAt) < HYST_MS;
        if ((bDist + 0) >= (cDist - HYST_PX) || recentlySwitched) {
          // Keep current to prevent flicker
          best = current;
        }
      }

      if (current && current !== best) current.classList.remove('is-focused');
      if (best && best !== current) {
        best.classList.add('is-focused');
        lastSwitchAt = Date.now();
      }
    };

    const schedule = () => {
      if (!scheduled) {
        scheduled = true;
        requestAnimationFrame(applyFocus);
      }
    };

    window.addEventListener('scroll', schedule, {
      passive: true
    });
    window.addEventListener('resize', schedule);

    // Re-evaluate when features enter/leave viewport using a lightweight observer
    const io = new IntersectionObserver(() => schedule(), {
      threshold: [0, 0.5, 1]
    });
    document.querySelectorAll('.feature').forEach(el => io.observe(el));

    // Initial run
    schedule();
  })();

  // Observe feature cards: reveal + apply in-view hover style while on screen
  const featureObserver = new IntersectionObserver((entries) => {
    entries.forEach(entry => {
      const el = entry.target;
      if (entry.isIntersecting) {
        el.classList.add('visible', 'in-view');
      } else {
        el.classList.remove('in-view', 'is-focused');
      }
    });
  }, {
    threshold: 0.15,
    rootMargin: '0px 0px -10% 0px'
  });
  document.querySelectorAll('.feature').forEach(feature => {
    featureObserver.observe(feature);
  });
}
