// js/hero.js

export function initHero() {
  // Ensure below-hero content is visible immediately on mobile and desktop
  document.body.classList.add('content-in');

  // Function to show CTA and scroll indicator
  const showHeroCtaAndScrollIndicator = () => {
    const heroCtaDiv = document.querySelector('.hero-cta-div');
    if (heroCtaDiv) {
      heroCtaDiv.classList.add('show');
    }
    // Delay showing the scroll indicator
    setTimeout(() => {
      document.body.classList.add('show-scroll-indicator');
      // Explicitly update scroll indicator after adding the class
      updateScrollIndicator(document.querySelector('.scroll-indicator'));
    }, 500);
  };

  // Immediately show CTA if hero is already typed (e.g., on page reload)
  if (document.body.classList.contains('hero-typed-done')) {
    showHeroCtaAndScrollIndicator();
  }

  // Listen for the hero-typed-done event to show CTA and scroll indicator
  window.addEventListener('hero-typed-done', showHeroCtaAndScrollIndicator);

  const hero = document.querySelector('.hero');
  const root = document.documentElement;
  const bodyEl = document.body;

  if (hero) {
    const updateHeroOverlay = () => {
      const rect = hero.getBoundingClientRect();
      const vh = window.innerHeight || 1;
      const heroHeight = Math.max(rect.height, vh);
      let scrolled = Math.min(Math.max((0 - rect.top) / heroHeight, 0), 1);
      const isMobile = (window.innerWidth || 0) <= 640;

      if (!isMobile) {
        const baseOverlay = 0.65;
        const extra = 0.25;
        const overlay = Math.min(0.95, baseOverlay + scrolled * extra);
        root.style.setProperty('--hero-overlay', overlay.toFixed(3));

        const fadeStart = 0.6, fadeEnd = 1.0;
        let t = 0;
        if (scrolled <= fadeStart) t = 0;
        else if (scrolled >= fadeEnd) t = 1;
        else t = (scrolled - fadeStart) / (fadeEnd - fadeStart);
        const videoOpacity = 1 - t;
        root.style.setProperty('--hero-video-opacity', videoOpacity.toFixed(3));
      } else {
        // Mobile: fixed values; no recompute on scroll to avoid relayout
        root.style.setProperty('--hero-overlay', '0.70');
        root.style.setProperty('--hero-video-opacity', '1.0');
      }
      // content-in is now set at init; avoid re-adding during scroll
    };

    updateHeroOverlay();

    let ticking2 = false;
    window.addEventListener('scroll', () => {
      const isMobile = (window.innerWidth || 0) <= 640;
      if (isMobile) {
        // Do not touch overlay/video on mobile during scroll
        if (!document.body.classList.contains('content-in')) {
          document.body.classList.add('content-in');
        }
        return;
      }
      if (!ticking2) {
        ticking2 = true; // ensure only one rAF callback per frame
        window.requestAnimationFrame(() => {
          updateHeroOverlay();
          ticking2 = false;
        });
      }
    }, { passive: true });

    window.addEventListener('resize', () => {
      const isMobile = (window.innerWidth || 0) <= 640;
      if (!isMobile) updateHeroOverlay();
    });

    window.addEventListener('orientationchange', () => {
      const isMobile = (window.innerWidth || 0) <= 640;
      if (!isMobile) updateHeroOverlay();
    });
  } else {
    bodyEl.classList.add('content-in');
  }

  // Try to play background video without throwing
  const heroVideo = document.getElementById('heroVideo');
  if (heroVideo) {
    const p = heroVideo.play();
    if (p && typeof p.catch === 'function') {
      p.catch(() => {});
    }
  }

  const tagline = document.querySelector('.tagline');
  if (tagline) {
    setTimeout(() => tagline.classList.add('show'), 300);
  }

  // Typewriter effect for hero (DE/EN)
  (() => {
    // Prevent multiple initializations (e.g., if scripts are injected twice)
    if (window.__typedHeroInit) return;
    window.__typedHeroInit = true;

    const container = document.getElementById('typewriter-container');
    const content = container && container.querySelector('#typewriter-content');
    const out = container && container.querySelector('#typed-output');
    if (!container || !content || !out) {
      return;
    }

    // Build line list once
    const lines = [];
    content.querySelectorAll('h1,h2').forEach(el => {
      const link = el.querySelector('a');
      const text = (link ? link.textContent : el.textContent || '').trim();
      const href = link ? link.getAttribute('href') : null;
      lines.push({
        tag: el.tagName.toLowerCase(),
        text,
        href
      });
    });

    // Reset output and avoid layout thrash
    out.textContent = '';

    // If we've already typed this session, render instantly and skip animation
    try {
      if (window.sessionStorage && sessionStorage.getItem('heroTyped') === '1') {
        const clickables = new Set([
          'Transportroboter', 'Lieferroboter', 'Reinigungsroboter',
          'Transport Robots', 'Delivery Robots', 'Cleaning Robots'
        ]);
        for (const { tag, text, href } of lines) {
          const el = document.createElement(tag);
          if (tag === 'h1') el.classList.add('typewriter');
          let target = el;
          if (href || clickables.has(text)) {
            const a = document.createElement('a');
            a.href = href || '#roboter';
            a.className = 'robot-link';
            a.setAttribute('data-robot', text.toLowerCase().replace(/\s+/g, '-'));
            el.appendChild(a);
            target = a;
          }
          // Full text without typing cursor
          target.appendChild(document.createTextNode(text));

          // Small-screen manual break for the specific H1 phrase
          try {
            if (tag === 'h1' && (window.innerWidth || 0) <= 400) {
              const key = 'Personalsorgen?';
              const idx = (text || '').indexOf(key);
              if (idx !== -1) {
                target.textContent = '';
                const before = (text || '').slice(0, idx);
                const after = (text || '').slice(idx + key.length).replace(/^\s+/, '');
                if (before) target.appendChild(document.createTextNode(before));
                const span = document.createElement('span');
                span.textContent = key;
                target.appendChild(span);
                target.appendChild(document.createElement('br'));
                if (after) target.appendChild(document.createTextNode(after));
              }
            }
          } catch (_) {}

          out.appendChild(el);
        }
        // Mark as done so dependent UI can react
        document.body.classList.add('hero-typed-done');
        try {
          window.dispatchEvent(new Event('hero-typed-done'));
        } catch (_) {}
        return; // Skip the typing logic below
      }
    } catch (_) {}

    const isMobile = (window.innerWidth || 0) <= 640;
    const speed = isMobile ? 40 : 32;
    const linePause = isMobile ? 600 : 500;
    const clickables = new Set([
      'Transportroboter', 'Lieferroboter', 'Reinigungsroboter',
      'Transport Robots', 'Delivery Robots', 'Cleaning Robots'
    ]);

    let i = 0;
    let currentTypingEl = null;
    let activeTimeout = null;
    let paused = false;

    function clearActiveTimeout() {
      if (activeTimeout) {
        clearTimeout(activeTimeout);
        activeTimeout = null;
      }
    }

    function typeLine() {
      if (i >= lines.length) {
        if (currentTypingEl) currentTypingEl.classList.remove('typing');
        clearActiveTimeout();
        try {
          document.body.classList.add('hero-typed-done');
          window.dispatchEvent(new Event('hero-typed-done'));
        } catch (_) {}
        try {
          if (window.sessionStorage) sessionStorage.setItem('heroTyped', '1');
        } catch (_) {}
        return;
      }

      const { tag, text, href } = lines[i];
      const el = document.createElement(tag);
      if (tag === 'h1') el.classList.add('typewriter');

      let target = el;
      if (href || clickables.has(text)) {
        const a = document.createElement('a');
        a.href = href || '#roboter';
        a.className = 'robot-link';
        a.setAttribute('data-robot', text.toLowerCase().replace(/\s+/g, '-'));
        el.appendChild(a);
        target = a;
      }
      out.appendChild(el);

      // Create a span for the text content and a separate span for the blinking cursor
      const textContentSpan = document.createElement('span');
      target.appendChild(textContentSpan);
      const cursorSpan = document.createElement('span');
      cursorSpan.classList.add('typing');
      target.appendChild(cursorSpan);

      // If there was a previous typing element, ensure its cursor is removed
      if (currentTypingEl) {
        currentTypingEl.classList.remove('typing');
      }
      currentTypingEl = cursorSpan;

      let j = 0;

      function step() {
        if (paused) {
          clearActiveTimeout();
          activeTimeout = setTimeout(step, 120);
          return;
        }
        if (j <= text.length) {
          textContentSpan.textContent = text.slice(0, j);
          j++;
          clearActiveTimeout();
          activeTimeout = setTimeout(step, speed);
        } else {
          cursorSpan.classList.remove('typing');
          // Forced break on tiny phones after "Personalsorgen?"
          try {
            if (tag === 'h1' && (window.innerWidth || 0) <= 400) {
              const key = 'Personalsorgen?';
              const idx = (text || '').indexOf(key);
              if (idx !== -1) {
                const before = (text || '').slice(0, idx);
                const after = (text || '').slice(idx + key.length).replace(/^\s+/, '');
                target.innerHTML = '';
                if (before) target.appendChild(document.createTextNode(before));
                const span = document.createElement('span');
                span.textContent = key;
                target.appendChild(span);
                target.appendChild(document.createElement('br'));
                if (after) target.appendChild(document.createTextNode(after));
              }
            }
          } catch (_) {}
          i++;
          clearActiveTimeout();
          activeTimeout = setTimeout(typeLine, linePause);
        }
      }
      step();
    }

    // Small initial delay to ensure fonts/layout are ready
    activeTimeout = setTimeout(typeLine, 300);

    // Pause typing when hero is offscreen
    try {
      const heroSection = document.querySelector('.hero');
      if (heroSection && 'IntersectionObserver' in window) {
        const io = new IntersectionObserver((entries) => {
          entries.forEach((entry) => {
            paused = !entry.isIntersecting;
          });
        }, { threshold: 0.05 });
        io.observe(heroSection);
      }
    } catch (_) {}

    // Stop typing if user navigates away (e.g., SPA-like behavior)
    window.addEventListener('beforeunload', clearActiveTimeout);
  })();

  // Function to update scroll indicator visibility
  const updateScrollIndicator = (dot) => {
    if (!dot) return;
    const y = window.scrollY || window.pageYOffset || 0;
    const nearTop = y < 8; // small threshold to avoid flicker
    const showIndicator = document.body.classList.contains('show-scroll-indicator');
    dot.classList.toggle('hidden', !(showIndicator && nearTop));
  };

  // Hide scroll indicator as soon as user starts scrolling
  (() => {
    const dot = document.querySelector('.scroll-indicator');
    if (!dot) return;
    updateScrollIndicator(dot); // Initial update
    window.addEventListener('scroll', () => updateScrollIndicator(dot), { passive: true });
    window.addEventListener('resize', () => updateScrollIndicator(dot));
  })();
}
