export function initNavigation() {
  const hamburger = document.getElementById('hamburger');
  const nav = document.querySelector('header nav');

  if (!nav || !hamburger) {
    return;
  }

  const dropdowns = nav.querySelectorAll('.nav-dropdown');

  // --- State Management ---
  const isMobile = () => window.matchMedia('(max-width: 1072px)').matches;

  const closeMenu = () => {
    document.body.classList.remove('nav-open');
    hamburger.classList.remove('active');
    hamburger.setAttribute('aria-expanded', 'false');
    nav.querySelectorAll('.nav-dropdown.dd-open').forEach(dd => {
      dd.classList.remove('dd-open');
    });
  };

  const toggleMenu = () => {
    const isOpen = document.body.classList.toggle('nav-open');
    hamburger.classList.toggle('active', isOpen);
    hamburger.setAttribute('aria-expanded', isOpen ? 'true' : 'false');
  };

  // --- Event Listeners ---

  // Hamburger is special, it toggles and stops propagation to prevent the body listener from closing the menu right away.
  hamburger.addEventListener('click', (e) => {
    e.stopPropagation();
    toggleMenu();
  });

  // A single, global click handler to manage all navigation interactions
  document.body.addEventListener('click', (e) => {
    const inOverlay = document.body.classList.contains('nav-open');

    // Case 1: Menu is open and the click is inside the <nav> container
    if (inOverlay && nav.contains(e.target)) {
      const link = e.target.closest('a');
      if (!link) return; // Ignore clicks inside nav that are not on links

      const href = link.getAttribute('href');

      // Sub-case 1a: It's a dropdown trigger
      const dropdown = link.closest('.nav-dropdown');
      const isTrigger = dropdown && link.classList.contains('nav-link');
      if (isTrigger) {
        e.preventDefault();
        dropdown.classList.toggle('dd-open');
        // Close other dropdowns
        dropdowns.forEach(dd => {
          if (dd !== dropdown) dd.classList.remove('dd-open');
        });
        return; // Don't close the main menu
      }

      // Sub-case 1b: It's a modal link
      if (href && href.startsWith('#') && href.includes('Modal')) {
        e.preventDefault();
        const modalId = href.substring(1);
        const modal = document.getElementById(modalId);
        closeMenu();
        if (modal) modal.style.display = 'flex';
        return;
      }

      // Sub-case 1c: It's a same-page anchor link
      if (link.pathname === window.location.pathname && link.hash) {
        e.preventDefault();
        const target = document.querySelector(link.hash);
        closeMenu();
        if (target) {
          setTimeout(() => {
            target.scrollIntoView({ behavior: 'smooth' });
            // After scroll, focus the feature card if applicable
            setTimeout(() => {
              const card = target.classList.contains('feature') ? target : (target.closest && target.closest('.feature'));
              if (card) {
                document.querySelectorAll('.feature.is-focused').forEach(n => n.classList.remove('is-focused'));
                card.classList.add('is-focused');
                try { window.__featureUserFocusUntil = Date.now() + 1500; } catch (_) {}
              }
            }, 250);
          }, 250);
        }
        return;
      }

      // For all other links, we let the browser do its thing, but we close the menu first.
      closeMenu();
    }

    // Case 2: Menu is open and the click is outside the nav and hamburger
    if (inOverlay && !nav.contains(e.target) && e.target !== hamburger) {
      closeMenu();
    }
  });
  
  // --- Desktop & Resize Behavior ---
  window.addEventListener('resize', () => {
    if (!isMobile()) closeMenu();
  });

  if (!isMobile()) closeMenu();
  
  nav.querySelectorAll('.dropdown-content a').forEach(link => {
    link.addEventListener('click', (e) => {
      if (isMobile()) {
        // This is already handled by the body listener, but we need to stop the desktop-only logic
        return;
      }
      // Desktop hover-hiding logic
      const dd = link.closest('.nav-dropdown');
      if (!dd) return;
      dd.classList.add('dd-force-hide');
      setTimeout(() => dd.classList.remove('dd-force-hide'), 500);
    });
  });
}
