// js/utils.js

export function initUtils() {
  // On mobile, avoid browser restoring scroll (prevents initial top then jump)
  (() => {
    const isMobile = (window.innerWidth || 0) <= 640;
    if (isMobile && 'scrollRestoration' in history) {
      try {
        history.scrollRestoration = 'manual';
      } catch (e) {}
    }
    // If URL has a hash on reload, only strip it when the target anchor does not exist
    if (isMobile && location.hash) {
      try {
        const id = location.hash.slice(1);
        if (!document.getElementById(id)) {
          history.replaceState(null, '', location.pathname + location.search);
        }
      } catch (e) {}
    }
    // Ensure we stay at top on bfcache restores
    window.addEventListener('pageshow', e => {
      if (isMobile && e.persisted) {
        window.scrollTo(0, 0);
      }
    });
  })();

  // Robust mobile viewport height variable (fallback when dvh is flaky)
  (() => {
    let vhFrozen = false;
    const setVH = () => {
      if (vhFrozen) return;
      const vh = window.innerHeight * 0.01; // 1% of viewport height
      document.documentElement.style.setProperty('--app-vh', `${vh}px`);
    };
    setVH();
    window.addEventListener('resize', setVH);
    window.addEventListener('orientationchange', () => {
      vhFrozen = false;
      setVH();
    });
    // Freeze after first scroll on mobile to keep hero height stable
    window.addEventListener('scroll', () => {
      const isMobile = (window.innerWidth || 0) <= 640;
      if (isMobile && !vhFrozen) {
        vhFrozen = true;
      }
    }, { passive: true });
  })();

  

  // Set footer offset
  (function setFooterOffset() {
    const footerEl = document.querySelector('footer, .footer');

    function update() {
      const rect = footerEl ? footerEl.getBoundingClientRect() : null;
      const h = rect ? rect.height : 80;
      const px = Math.round(h + 16);
      document.documentElement.style.setProperty('--footer-offset', px + 'px');
    }
    update();
    window.addEventListener('resize', update);
    window.addEventListener('orientationchange', update);
  })();
}
