<?php
// Include the internationalization functions
require_once __DIR__ . '/includes/i18n.php';
require_once __DIR__ . '/includes/smtp_mailer.php';
// Optional app config (local dev)
$APP_CFG = [];
$cfgPath = __DIR__ . '/includes/config.php';
if (file_exists($cfgPath)) {
    $loaded = include $cfgPath;
    if (is_array($loaded)) { $APP_CFG = $loaded; }
}

// Lightweight .env loader (root-level .env) – PHP 7 compatible
(function($root){
    $envFile = $root . '/.env';
    if (!is_readable($envFile)) return;
    $lines = @file($envFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    if (!$lines) return;
    foreach ($lines as $line) {
        $line = trim($line);
        if ($line === '' || $line[0] === '#' || strpos($line, '=') === false) continue;
        [$k, $v] = array_map('trim', explode('=', $line, 2));
        if ($k === '') continue;
        // Strip surrounding quotes (no str_starts_with/ends_with in PHP7)
        $len = strlen($v);
        if ($len >= 2) {
            $first = $v[0];
            $last  = $v[$len-1];
            if (($first === '"' && $last === '"') || ($first === "'" && $last === "'")) {
                $v = substr($v, 1, -1);
            }
        }
        // Expand simple ${VAR} references
        $v = preg_replace_callback('/\$\{([A-Z0-9_]+)\}/i', function($m){
            $ref = getenv($m[1]);
            return $ref !== false ? $ref : '';
        }, $v);
        putenv($k . '=' . $v);
        $_ENV[$k] = $v;
        $_SERVER[$k] = $v;
    }
})(__DIR__);

if ($_SERVER["REQUEST_METHOD"] == "POST") {
    // Helper: read environment variable from multiple sources
    $env = function(string $key, $default = '') {
        $v = getenv($key);
        if ($v === false || $v === null || $v === '') {
            $v = $_ENV[$key] ?? $_SERVER[$key] ?? $default;
        }
        return $v;
    };
    // Lightweight logger with multiple fallbacks
    $logMailFailure = function(string $line): void {
        $line = rtrim($line, "\n") . "\n";
        // 1) Try logs/ directory (attempt to create if missing)
        $dir1 = __DIR__ . '/logs';
        if (!is_dir($dir1)) {
            @mkdir($dir1, 0775, true);
        }
        $file1 = $dir1 . '/mail_failures.log';
        if (@file_put_contents($file1, $line, FILE_APPEND) !== false) {
            return;
        }
        // 2) Project root file
        $file2 = __DIR__ . '/mail_failures.log';
        if (@file_put_contents($file2, $line, FILE_APPEND) !== false) {
            return;
        }
        // 3) System temp dir
        $tmp = rtrim(sys_get_temp_dir(), '/');
        @file_put_contents($tmp . '/mail_failures.log', $line, FILE_APPEND);
        // 4) Finally, PHP error log
        @error_log('[send_mail.php] ' . trim($line));
    };
    // Determine recipient(s): config `mail.to` or environment `MAIL_TO` (comma/semicolon/space separated)
    $toConf = $APP_CFG['mail']['to'] ?? $env('MAIL_TO', '');
    if (is_array($toConf)) {
        $toList = $toConf;
    } else {
        $toStr = (string)$toConf;
        if ($toStr === '') {
            // Fallback to previous default
            $toStr = 'a.friesen@frisilogix.de';
        }
        $toList = preg_split('/[,;\s]+/', $toStr, -1, PREG_SPLIT_NO_EMPTY);
    }
    $toRecipients = array_values(array_unique(array_filter($toList, function($e) {
        return (bool)filter_var($e, FILTER_VALIDATE_EMAIL);
    })));
    if (empty($toRecipients)) {
        $error = 'No valid recipient configured (use mail.to in config.php or MAIL_TO env).';
        $logMailFailure('[FATAL] ' . $error);
        http_response_code(500);
        echo 'MAIL_ERROR: ' . $error;
        exit;
    }

    // Safely get and sanitize form data
    $name = strip_tags(trim($_POST["name"] ?? ''));
    $email = filter_var(trim($_POST["email"] ?? ''), FILTER_SANITIZE_EMAIL);
    $subject = strip_tags(trim($_POST["subject"] ?? ''));
    $message_body = trim($_POST["message"] ?? '');

    // Simple validation
    if (empty($name) || empty($subject) || empty($message_body) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
        http_response_code(400);
        echo "Bitte füllen Sie alle Felder korrekt aus und geben Sie eine gültige E-Mail an.";
        exit;
    }

    // Compose email content
    $email_content = "Name: $name\n";
    $email_content .= "E-Mail: $email\n\n";
    $email_content .= "Betreff: $subject\n";
    $email_content .= "--------------------------------------\n\n";
    $email_content .= "Nachricht:\n$message_body\n";

    // Try SMTP when configured via env/config; else store to outbox
    $smtpHost   = (string)($APP_CFG['mail']['smtp']['host'] ?? $env('SMTP_HOST', ''));
    $smtpPort   = (int)   ($APP_CFG['mail']['smtp']['port'] ?? $env('SMTP_PORT', 465));
    $smtpUser   = (string)($APP_CFG['mail']['smtp']['user'] ?? $env('SMTP_USER', ''));
    $smtpPass   = (string)($APP_CFG['mail']['smtp']['pass'] ?? $env('SMTP_PASS', ''));
    $smtpSecure = strtolower((string)($APP_CFG['mail']['smtp']['secure'] ?? $env('SMTP_SECURE', 'ssl')));
    $smtpFrom   = (string)($APP_CFG['mail']['smtp']['from'] ?? $env('SMTP_FROM', ''));
    $smtpFromName = (string)($APP_CFG['mail']['smtp']['from_name'] ?? 'Kontaktformular');
    $smtpDebug  = (bool)  ($APP_CFG['mail']['smtp']['debug'] ?? (bool)$env('SMTP_DEBUG', ''));

    $hasSmtp = ($smtpHost !== '' && $smtpFrom !== '');
    if ($hasSmtp) {
        $domain = preg_replace('/^www\./i', '', $smtpHost);
        $cfg = [
            'host'       => $smtpHost,
            'port'       => $smtpPort,
            'user'       => $smtpUser,
            'pass'       => $smtpPass,
            'secure'     => $smtpSecure,
            'timeout'    => 15,
            'msgid_host' => $domain,
            'ehlo'       => $domain,
            'debug'      => $smtpDebug,
        ];

        $fromNameForHeader = $name;
        $encodedName = mime_header_encode($name);
        $headers = [ 'Reply-To' => '"' . $encodedName . '" <' . $email . '>' ];

        [$ok, $err] = smtp_send($cfg, $smtpFrom, $fromNameForHeader, $toRecipients, 'Neue Kontaktanfrage: ' . $subject, $email_content, $headers);
        if ($ok) {
            $successLine = sprintf('[%s] smtp_send ok: to=%s from=%s user=%s subject=%s', date('c'), implode(',', $toRecipients), $smtpFrom, ($cfg['user'] ? 'set' : 'none'), 'Neue Kontaktanfrage: ' . $subject);
            $succDir = __DIR__ . '/logs';
            if (!is_dir($succDir)) { @mkdir($succDir, 0775, true); }
            @file_put_contents($succDir . '/mail_sent.log', $successLine . "\n", FILE_APPEND);

            http_response_code(200);
            echo 'OK';
            exit;
        }

        // Log failure and fall through to outbox store
        $logMailFailure(sprintf('[%s] smtp_send failed: %s host=%s port=%s secure=%s user=%s', date('c'), (string)$err, $smtpHost, (string)$smtpPort, $smtpSecure, $smtpUser !== '' ? 'set' : 'none'));
    }

    // Outbox store (fallback or when SMTP is not configured)
    $replyTo = $email;
    $fromHeader = $name . ' <no-reply@frisilogix.de>';
    $headersText = "To: " . implode(', ', $toRecipients) . "\n"
                 . "From: " . $fromHeader . "\n"
                 . "Reply-To: \"" . $name . "\" <" . $replyTo . ">\n"
                 . "Subject: Neue Kontaktanfrage: " . $subject . "\n"
                 . "Date: " . date('r') . "\n"
                 . "Content-Type: text/plain; charset=UTF-8\n";

    $out = $headersText . "\n" . $email_content . "\n";
    $outDir = __DIR__ . '/logs/outbox';
    if (!is_dir($outDir)) { @mkdir($outDir, 0775, true); }
    if (!is_writable($outDir)) {
        $alt = __DIR__ . '/logs';
        if (!is_dir($alt)) { @mkdir($alt, 0775, true); }
        if (is_writable($alt)) { $outDir = $alt; }
    }
    $file = $outDir . '/contact-' . date('Ymd-His') . '-' . bin2hex(random_bytes(3)) . '.txt';
    @file_put_contents($file, $out);

    $successLine = sprintf('[%s] contact stored (no SMTP): to=%s file=%s subject=%s', date('c'), implode(',', $toRecipients), basename($file), 'Neue Kontaktanfrage: ' . $subject);
    $succDir = __DIR__ . '/logs';
    if (!is_dir($succDir)) { @mkdir($succDir, 0775, true); }
    @file_put_contents($succDir . '/mail_sent.log', $successLine . "\n", FILE_APPEND);

    http_response_code(200);
    echo 'OK';
    exit;

} else {
    // Block direct access to the script
    http_response_code(403);
    echo "Zugriff verweigert.";
}
?>
